/******************************************************************************
													Copyright (c) Freescale 2009
File Name		 :	$RCSfile: EE_emulation.c,v $
	
Engineer		 :	$Author: b21465 $

Location		 :	Brazil

Date Created	 :	15/09/2009

Current Revision :	$Revision: 1.0 $

Notes	:					

*******************************************************************************
Freescale reserves the right to make changes without further notice to any
product	herein to improve reliability, function	or design. Freescale does not
assume any liability arising out of the	application or use of any product,
circuit, or software described	herein;	neither	does it	convey any license
under its patent rights	nor the rights of others. Freescale products are	not
designed, intended, or authorized for use as components in systems intended for
surgical implant into the body, or other applications intended to support life,
or for any other application in which the failure of the  Freescale product
could create a	situation where	personal injury	or death may occur. Should
Buyer purchase or use Freescale products for any such unintended or
unauthorized application, Buyer shall idemnify and hold Freescale and its
officers, employees, subsidiaries, affiliates, and distributors	harmless
against	all claims costs, damages, and expenses, and reasonable	attorney fees 
arising	out of,	directly or indirectly,	any claim of personal injury or	death
associated with	such unintended	or unauthorized use, even if such claim alleges
that Freescale was negligent regarding the design or manufacture	of the part.
Freescale and the Freescale logo are registered trademarks of Freescale Ltd.
******************************************************************************/

/************************* System Include Files ******************************/

/************************* Project Include Files *****************************/

#include "EE_AN_emulation.h"
#include "ee_emulation.h"

/************************* #Defines ******************************************/
								/* number of WORDS of emulated EEPROM, including status word */
#define EEPROM_SIZE_WORDS		((EEPROM_SIZE_BYTES + 2) / 2)
						  	/* number of WORDS in a flash erase sector - DFLASH_SECTOR_SIZE = number of bytes*/
#define FLASH_SECTOR_WORDS 		(DFLASH_SECTOR_SIZE/2)
								/* erased flash word */
#define FLASH_ERASED_WORD		0xFFFFu
								/* erased flash byte */
#define FLASH_ERASED_BYTE		0xFFu

								/* address mask for aligned word */
#define ALIGNED_WORD_MASK		0xFFFFFEu
								/* address mask for start of flash sector */
#define FLASH_SECTOR_MASK		0xFFFF00u
								/* Mask for I flag in CCR. */
#define IMASK					0x10

								/* Parameter checking */
#if(EEPROM_SIZE_WORDS < FLASH_SECTOR_WORDS)
#if((FLASH_SECTOR_WORDS % EEPROM_SIZE_WORDS) != 0)
/* EEPROM_SIZE_WORDS must equal FLASH_SECTOR_WORDS / n */
#error Invalid value for EEPROM_SIZE_BYTES
#endif
#else  					/* (EEPROM_SIZE_WORDS >= FLASH_SECTOR_WORDS */
#if((EEPROM_SIZE_WORDS % FLASH_SECTOR_WORDS) != 0)
/* EEPROM_SIZE_WORDS must equal n * FLASH_SECTOR_WORDS */
#error Invalid value for EEPROM_SIZE_BYTES
#endif
#endif

#if(((EEPROM_SIZE_WORDS * EEPROM_BANKS) / FLASH_SECTOR_WORDS) < 2)
/* EEPROM_SIZE_WORDS * EEPROM_BANKS must equal (2 or more) * FLASH_SECTOR_WORDS */
#error Invalid value for EEPROM_SIZE_BYTES or EEPROM_BANKS
#endif

#if(((EEPROM_SIZE_WORDS * EEPROM_BANKS) % FLASH_SECTOR_WORDS) != 0)
/* EEPROM_SIZE_WORDS * EEPROM_BANKS must equal n * FLASH_SECTOR_WORDS */
#error Invalid value for EEPROM_SIZE_BYTES or EEPROM_BANKS
#endif


#if((EEPROM_SIZE_WORDS * EEPROM_BANKS) > (DFLASH_MAXIMUM_SIZE))
/* EEPROM_SIZE_WORDS * EEPROM_BANKS must equal n * FLASH_SECTOR_WORDS */
#error Invalid value for EEPROM_SIZE_BYTES or EEPROM_BANKS
#endif

/************************* Prototypes ******************************************/

#pragma CODE_SEG DEFAULT
static UINT16 ProgEepromWord(UINT16 *SSD_SGF18_FAR*, UINT16);
static UINT16 EraseEepromBank(UINT16 *SSD_SGF18_FAR);


/************************* Extern Variables **********************************/

/************************* Global Variables **********************************/

#pragma DATA_SEG EEPROM_STAT
								/* EEPROM status, first word of each EEPROM bank */
static VUINT16 EepromStatus;
 
#pragma DATA_SEG DEFAULT
										
#if(EEPROM_BANKS < 256)
								/* eeprom bank currently in use */
static UINT8 ActiveBank;
#else
static UINT16 ActiveBank;
#endif


/************************* Functions *****************************************/
#pragma CODE_SEG DEFAULT
/******************************************************************************
Function Name:	ReadEeprom
Engineer:       b21465
Date:           15/09/2009

Parameters: srcAddr:	pointer to the non-volatile data variable to be read.
					  destAddr: pointer to a RAM location to copy the read value to.
					  size:		  size in bytes of the non-volatile data variable to be read.				 

Returns:    void

Notes:      The ReadEeprom function is used to obtain the current value of a 
            non-volatile data variable. This is the only way to obtain the 
            current value, accessing the data variable will NOT generally 
            result in the current value being read.
            InitEeprom must have been called once before this function 
					  is called.
******************************************************************************/
void 
ReadEeprom(void *SSD_SGF18_FAR srcAddr, void *destAddr, UINT16 size)
{
								/* current location */
	(UINT8*SSD_SGF18_FAR)srcAddr += (ActiveBank * (EEPROM_SIZE_BYTES + 2));

	while(size != 0)
	{
								/* copy to destination */
		*(UINT8*)destAddr = *(UINT8*SSD_SGF18_FAR)srcAddr;
								/* next address */
		((UINT8*)destAddr)++;
								/* next address */
		((UINT8*SSD_SGF18_FAR)srcAddr)++;
								/* one byte less */		
		size--;
	}
}

/******************************************************************************
Function Name:	WriteEeprom
Engineer:       b21465
Date:           15/09/2009

Parameters:	destAddr:	pointer to the non-volatile data variable to be updated.
					  srcAddr:	pointer to the new data to be written to the non-volatile 
					            data variable
					  size:     size in bytes of the non-volatile data variable to be updated				 

Return:	    FAIL:     a failure occurred during programming.
            PASS		update was successful.

Notes:      The WriteEeprom function is used to update a non-volatile data 
            variable with a new value
            InitEeprom must have been called once before this function is called. 
            Whenever this function is called, a whole new bank is programmed. 
            This will take approximately (EEPROM_SIZE_BYTES + 2) x 50us. 
            Furthermore, a Flash sector is erased if there are no erased banks 
            (multiple sectors if EEPROM_SIZE_WORDS > FLASH_SECTOR_WORDS), this 
            will take	approximately 20ms per sector.
            EepromStatus word (1st word in bank) is programmed only if programming 
            of all data is successfull. 
            Sector erase, if required, occurs before programming.
******************************************************************************/
UINT16 
WriteEeprom(void *SSD_SGF18_FAR destAddr, void *srcAddr, UINT16 size)
{
	UINT16  status;
#if(EEPROM_BANKS < 256)
	UINT8  oldBank;
#else
	UINT16 oldBank;
#endif
#if(EEPROM_SIZE_WORDS < 256)
	UINT8  count;
#elif(EEPROM_SIZE_WORDS < FLASH_SECTOR_WORDS)
	UINT16 count;
#endif
	UINT16 *SSD_SGF18_FAR oldBankAddr;
	UINT16 *SSD_SGF18_FAR eepromAddr;
	union U16  	/*16 bit data with word and byte access*/
  	{
  		UINT16  word;           		
 		struct                			
    {
    	UINT8  msb;
    	UINT8  lsb;
    }byte;
	}eepromData;

								/* check for valid address, exclude status word */
	if(((UINT16 *SSD_SGF18_FAR)destAddr < (UINT16*)((UINT16 *SSD_SGF18_FAR)EEPROM_START + 2)) ||
	   ((UINT16 *SSD_SGF18_FAR)destAddr > (UINT16*)((UINT16 *SSD_SGF18_FAR)EEPROM_START + EEPROM_SIZE_BYTES + 2 - size)))
	{
		return(FAIL);
	}
								/* start of old bank, excl status word */
	oldBank = ActiveBank;
	oldBankAddr = (UINT16 *SSD_SGF18_FAR)EEPROM_START + (ActiveBank * EEPROM_SIZE_WORDS) + 1;
										
	/* find next erased bank, erase sector if required */
	do {
		status = PASS;
								/* move to new bank */
		ActiveBank++;
								/* check for end of banks */
		if(ActiveBank == EEPROM_BANKS)
		{
								/* back to start */
			ActiveBank = 0;
		}
								/* start address of new bank */
		eepromAddr = (UINT16 *SSD_SGF18_FAR)EEPROM_START + (ActiveBank * EEPROM_SIZE_WORDS);
								/* multiple banks per sector */
#if(EEPROM_SIZE_WORDS < FLASH_SECTOR_WORDS)
								/* start of new sector? */
		if((UINT32)eepromAddr == ((UINT32)eepromAddr & FLASH_SECTOR_MASK))
#endif
		{
								/* erase new bank sector(s) */
			status = EraseEepromBank(eepromAddr);
			
			if(status != PASS)
			{					/* failed to erase */
				ActiveBank = oldBank;
				return(FAIL);
			}
		}
								/* multiple banks per sector */
#if(EEPROM_SIZE_WORDS < FLASH_SECTOR_WORDS)
								/* not a new sector */
		else							
		{
								/* Is this bank erased? */
			for(count = 0; count < EEPROM_SIZE_WORDS; count++)
			{	
				if(*eepromAddr++ != FLASH_ERASED_WORD)
				{
								/* not erased */
					status = FAIL;
					break;
				}
			}
		}
#endif
									
	}while(status == FAIL);
	
	/* program new bank */
  /* dest location in new bank */
	(UINT8 *SSD_SGF18_FAR)destAddr += (ActiveBank * (EEPROM_SIZE_BYTES + 2));
								/* start of bank to program, skip status word */
	eepromAddr = (UINT16 *SSD_SGF18_FAR)EEPROM_START + (ActiveBank * EEPROM_SIZE_WORDS) + 1;
								/* copy first part of old bank */
	while(eepromAddr < (UINT16 *SSD_SGF18_FAR)((UINT32)destAddr & ALIGNED_WORD_MASK))
	{
								/* fetch word to copy */
		eepromData.word = *oldBankAddr;
								/* program location */
		status |= ProgEepromWord(&eepromAddr, eepromData.word);
								/* next data */
		oldBankAddr++;
	}
								/* first byte odd? */
	if(destAddr != eepromAddr)
	{
								/* fetch word to copy */
		eepromData.word = *oldBankAddr;
								/* overwrite with data byte */
		eepromData.byte.lsb = *(UINT8*)srcAddr;
								/* program location */
		status |= ProgEepromWord(&eepromAddr, eepromData.word);
								/* track old bank */
		oldBankAddr++;
								/* next data byte */
		((UINT8*)srcAddr)++;
								/* one byte less */
		size--;
	}
								/* program location with new data */
	while(size != 0)
	{
								/* single last byte? */
		if(size == 1)
		{							
								/* fetch word to copy */
			eepromData.word = *oldBankAddr;
								/* overwrite with data byte */
			eepromData.byte.msb = *(UINT8*)srcAddr;
								/* done */
			size--;
		}
								/* aligned word */
		else
		{
								/* fetch data word */
			eepromData.word = *(UINT16*)srcAddr;
								/* one word less */
			size -= 2;
		}
								/* program location */
		status |= ProgEepromWord(&eepromAddr, eepromData.word);
								/* track old bank */
		oldBankAddr++;
								/* next data word */
		((UINT16*)srcAddr)++;
	}
								/* copy last part of old bank */
	while(eepromAddr < (UINT16 *SSD_SGF18_FAR)EEPROM_START + ((ActiveBank + 1) * EEPROM_SIZE_WORDS))
	{
								/* fetch word to copy */
		eepromData.word = *oldBankAddr;
								/* program location */
		status |= ProgEepromWord(&eepromAddr, eepromData.word);
								/* next data */
		oldBankAddr++;
	}
								/* no errors? */
	if(status == PASS)
	{
								/* fetch status word of old bank */
		eepromData.word = *((UINT16 *SSD_SGF18_FAR)EEPROM_START + (oldBank * EEPROM_SIZE_WORDS));
								/* increment, skip erased state */
		if(eepromData.word++ == FLASH_ERASED_WORD)
		{
			eepromData.word = 0;
		}
								/* first word of bank is status word */
		eepromAddr -= EEPROM_SIZE_WORDS;
								/* program location */
		status |= ProgEepromWord(&eepromAddr, eepromData.word);
	}

	if(status != PASS)
	{	
										/* failed to program */
		ActiveBank = oldBank;
	}
	return(status);
}

/******************************************************************************
Function Name:	InitEeprom
Engineer:       b21465
Date:           15/09/2009

Parameters:	void
Return:	    void

Notes:      This function must be called once before ReadEeprom or WriteEeprom 
            are called. This function initialises the Flash prescaler, copies 
            the required functions to RAM and determines the bank with the most 
            recent data from the bank EepromStatus word. If no valid data 
            is found, bank 0 is erased - this takes	20ms.
******************************************************************************/
void
InitEeprom(void)
{
  UINT16 returnCode;      /* return code variable */

#if(EEPROM_BANKS < 256)
	UINT8  bank;
#else
	UINT16 bank;
#endif

	UINT16 statusWord;
	UINT16 largestStatus = 0;
	UINT8  erased = TRUE;
	UINT8  statusRollOver = FALSE;

  returnCode = FlashInit();
  
					/* search all banks for largest Status value */
	for(bank = 0; bank < EEPROM_BANKS; bank++)
	{
	  /* status word at start of each bank */
		statusWord = *((UINT16 *SSD_SGF18_FAR)EEPROM_START + (bank * EEPROM_SIZE_WORDS));
    
		if(statusWord != FLASH_ERASED_WORD)
		{
			erased = FALSE;
			
			if(statusRollOver == FALSE)
			{							
				if(statusWord == 0)
				{						/* possible EepromStatus word roll-over */
					statusRollOver = TRUE;	
										/* search again, starting with bank 0 */				
#if(EEPROM_BANKS < 256)
					bank = 0xFF;
#else
					bank = 0xFFFF;
#endif
					largestStatus = 0;
				}

				else if(statusWord > largestStatus)
				{						/* largest so far */
					ActiveBank = bank;
					largestStatus = statusWord;
				}

			}
			else
			{							/* EepromStatus roll-over - 2nd search */
				if((statusWord < EEPROM_BANKS) && (statusWord > largestStatus))
				{
										/* largest < EEPROM_BANKS so far */
					ActiveBank = bank;
					largestStatus = statusWord;
				}
			}
		}
	}
										
	if(erased == TRUE)
	{									/* all status words erased - no valid data */
										/* bank 1 will be programmed first */
		ActiveBank = 0;
										/* erase bank 0 sector(s) */
		(void)EraseEepromBank((UINT16*SSD_SGF18_FAR)EEPROM_START);
	}
										/* dummy read so linker includes EepromStatus */
	statusWord = EepromStatus;
}

/******************************************************************************
Function Name	:	ProgEepromWord
Engineer		:	r27624
Date			:	20/01/2003

Parameters	    :	progAddr: pointer to pointer containing flash address
					
					data	: data word to be programmed	 

Returns		    :	status	: PASS if programming successful
							: FAIL if programming unsuccessful

Notes			:	flash address in *progAddr is incremented by this routine
******************************************************************************/
UINT16
ProgEepromWord(UINT16 *SSD_SGF18_FAR* progAddr, UINT16 data)
{
	UINT16 status;
	
  UINT32 address;

  address = (UINT32)(*progAddr);
  
  status = FSL_Program ((UINT32)*progAddr, 2, data);
	/* program location */
	
	/* verify */
	if(**progAddr != data)
	{
		status = FAIL;
	}
										/* next location */
	*progAddr += 1;

	return(status);	
}

/******************************************************************************
Function Name:	EraseEepromBank
Engineer:       b21465
Date:           15/09/2009

Parameters:	eepromAddr: pointer containing start address of EEPROM bank
Return:     status:     PASS: if ProgFlash returns PASS
							          FAIL: if ProgFlash returns FAIL

Notes:      This will take around 20ms per sector erased.
******************************************************************************/
UINT16
EraseEepromBank(UINT16 *SSD_SGF18_FAR eepromAddr)
{
	UINT16 status = 0;
	UINT32 address;
	
  address = (UINT32)((UINT16 *SSD_SGF18_FAR)EEPROM_START + ((ActiveBank + 1) * EEPROM_SIZE_WORDS));
  
										/* multiple sectors per bank */
#if(EEPROM_SIZE_WORDS > FLASH_SECTOR_WORDS)
  for( ; 
	  /* end of bank? */
	  eepromAddr < (UINT16 *SSD_SGF18_FAR)EEPROM_START + ((ActiveBank + 1) * EEPROM_SIZE_WORDS);
	  /* next sector */
	  eepromAddr += FLASH_SECTOR_WORDS)
#endif									/* else EEPROM_SIZE_WORDS <= FLASH_SECTOR_WORDS */
	  {
  		/* erase sector */
      status |= DFlashErase((UINT32)(eepromAddr), 2 * FLASH_SECTOR_WORDS);
  	}
  return(status);
}
